function commonSubmitArgs = getCommonSubmitArgs(cluster, numWorkers)
% Get any additional submit arguments for the Slurm sbatch command
% that are common to both independent and communicating jobs.

% Copyright 2016-2022 The MathWorks, Inc.

% wiki:

commonSubmitArgs = '';

% Number of cores/node
ppn = validatedPropValue(cluster, 'ProcsPerNode', 'double');
if ppn>0
    % Don't request more cores/node than workers
    ppn = min(numWorkers*cluster.NumThreads,ppn);
    assert(rem(ppn,cluster.NumThreads)==0, ...
        'ProcsPerNode (%d) must be greater than or equal to and divisible by NumThreads (%d).', ppn, cluster.NumThreads);
    commonSubmitArgs = sprintf('%s --ntasks-per-node=%d',commonSubmitArgs,ppn/cluster.NumThreads);
end
commonSubmitArgs = sprintf('%s --ntasks-per-core=1',commonSubmitArgs);

%% REQUIRED

% qn = validatedPropValue(cluster, 'QueueName', 'char');
% if isempty(qn)
%     emsg = sprintf(['\n\t>> %% Must set QueueName.  E.g.\n\n', ...
%                     '\t>> c = parcluster;\n', ...
%                     '\t>> c.AdditionalProperties.QueueName = ''queue-name'';\n', ...
%                     '\t>> c.saveProfile\n\n']);
%     error(emsg)
% else
%     commonSubmitArgs = sprintf('%s -p %s', commonSubmitArgs, qn);
% end

%% OPTIONAL

% Account name
an = validatedPropValue(cluster, 'AccountName', 'char');
if ~isempty(an)
    commonSubmitArgs = sprintf('%s -A %s', commonSubmitArgs, an);
end

% Constraint
cn = validatedPropValue(cluster, 'Constraint', 'char');
if ~isempty(cn)
    commonSubmitArgs = sprintf('%s -C %s', commonSubmitArgs, cn);
end

% Queue name
qn = validatedPropValue(cluster, 'QueueName', 'char');
if ~isempty(qn)
    commonSubmitArgs = sprintf('%s -p %s', commonSubmitArgs, qn);
end

% GPU
ngpus = validatedPropValue(cluster, 'GpusPerNode', 'double');
if ngpus>0
    commonSubmitArgs = sprintf('%s --gres=gpu:%d', commonSubmitArgs, ngpus);
end

% Wall time
wt = validatedPropValue(cluster, 'WallTime', 'char');
if ~isempty(wt)
    commonSubmitArgs = sprintf('%s -t %s', commonSubmitArgs, wt);
end

% Physical memory used by a single core
mu = validatedPropValue(cluster, 'MemUsage', 'char');
if ~isempty(mu)
    commonSubmitArgs = sprintf('%s --mem-per-cpu=%s', commonSubmitArgs, mu);
end

% Reservation
res = validatedPropValue(cluster, 'Reservation', 'char');
if ~isempty(res)
    commonSubmitArgs = sprintf('%s --reservation=%s', commonSubmitArgs, res);
end

% Require exclusive node
ren = validatedPropValue(cluster, 'RequireExclusiveNode', 'bool');
if ren
    commonSubmitArgs = sprintf('%s --exclusive', commonSubmitArgs);
end

% Email notification
ea = validatedPropValue(cluster, 'EmailAddress', 'char');
if ~isempty(ea)
    commonSubmitArgs = sprintf('%s --mail-type=ALL --mail-user=%s', commonSubmitArgs, ea);
end

% Catch-all
asa = validatedPropValue(cluster, 'AdditionalSubmitArgs', 'char');
if ~isempty(asa)
    commonSubmitArgs = sprintf('%s %s', commonSubmitArgs, asa);
end

commonSubmitArgs = strtrim(commonSubmitArgs);

end
